# psx-acars: program to link Hoppie's ACARS into PSX as the ACARS system on
# the MCDUs.
# https://www.hoppie.nl/

# Jeroen Hoppenbrouwers <hoppie@hoppie.nl> September 2025.

VERSION = "beta-03"

import aiohttp
import asyncio
from datetime import datetime # use dt for the modified variant
from datetime import timezone
from datetime import timedelta
from enum import Enum
from math import ceil
import modules.psx
import modules.acars
import os
import sys
import textwrap as tw
import time
import traceback


##### Global variables ###################################################

ZULU = None


##### Classes and Functions ##############################################

class dt(datetime):
  """Subclass of datetime to override some default methods."""

  def __str__(self):
    """When converted to default string, replace UTC time zone with Zulu."""
    if self.tzname()=="UTC":
      return self.strftime("%Y-%b-%d %H:%M:%SZ")
    else:
      return self.strftime("%Y-%b-%d %H:%M:%S%z")
  # str()

# dt


def setup():
  # This is a synchronous (normal) function, so don't use await.
  # The only things you can do are changing flags and such that get picked
  # up by truly async coroutines/tasks.
  print("ACARS setup")
  time.sleep(2)   # Give PSX some time, needed after a situ reload and resume.
  # Display we're in NO COMM but do not yet flag the system as broken.
  psx.send("FreeMsgC", "")
  psx.send("FreeMsgA", ">NO COMM")
  psx.send("FreeMsgM", "")
  psx.send("FreeMsgS", "")
  psx.send("AcarsMsg", "0")

  if not stealth:
    mcdu_l.plugin_to(psx)
    mcdu_c.plugin_to(psx)
    mcdu_r.plugin_to(psx)

  # Set PSX FANS to "application control" and "logged off".
  psx.send("FansCont", "4")
  timer, _, rest = psx.get("FansBasics").partition(";")
  psx.send("FansBasics", f"0;{rest}")
  # psx.send("FansBasics",
  #    "0;;;e;2;0;0;;-1;622;31000;;;;;;;;;;-9999;;0;0;0;r;0;0;;;b;;;;;;;0;")
  print("End ACARS setup")
# setup();


def teardown(reason=None):
  # This is a synchronous (normal) function, so don't use await.
  # The only things you can do are changing flags and such that get picked
  # up by truly async coroutines/tasks.
  print(f"ACARS teardown ({reason})")
  # PSX FANS back to "manual control" and "logged off".
  FANS.logoff()
  time.sleep(1)   # Allow PSX to process the logoff.
  psx.send("FansCont", "0")
  # Remove all our potential EICAS messages.
  psx.send("FreeMsgC", "")
  psx.send("FreeMsgA", "")
  psx.send("FreeMsgM", "")
  psx.send("FreeMsgS", "")
  psx.send("AcarsMsg", "0")
  if not stealth:
    mcdu_l.unplug()
    mcdu_c.unplug()
    mcdu_r.unplug()

  AOC.callsign = None
  FANS.curr_atc = None
  FANS.next_atc = None

  print("End ACARS teardown")
# teardown()


class Message:
  """A single ACARS message."""

  def __init__(self, orig, typp, text):
    global ZULU
    self.orig   = orig
    self.type   = typp
    self.text   = text.upper()
    self.zulu   = ZULU
    self.is_new = True
    print(f"Stored new {typp} message from {orig}: {text[0:40]}")
  # init()

  def __str__(self):
    time = self.zulu.strftime("%H:%Mz")
    return f"{time} {self.orig}: {self.text}"
  # str()

# Message


class ScratchPad:
  # Handles all keys that are meaningful for the scratch pad.
  # TODO: scratch pad message stack, special pseudo keys "clear", "put"
  # and "get" for erase, downselect and upselect.

  # Scratch pad needs to paint itself, but cannot know which head to use for
  # this unless we give it the reference. Use app.mcdu.paint()
  def __init__(self, app):
    self.app    = app
    self.buffer = ""
    self.delete = False
  # init()


  def key(self, value):
    if value=="SP":
      value = " "
    elif value=="+/-":
      # Simplified to follow PSX system. Repeat push toggles symbol.
      if self.buffer[-1:]=="+":
        self.buffer = self.buffer[:-1]
        value = "-"
      elif self.buffer[-1:]=="-":
        self.buffer = self.buffer[:-1]
        value = "+"
      else:
        value = "-"

    if value=="CLR":
      self.buffer = self.buffer[:-1]
    elif value=="CLR+":
      self.buffer = ""
    elif value=="DEL":
      self.delete = not self.delete
    else:
      # Don't accept a space as first character.
      if value!=" " or self.buffer!="":
        self.buffer += value
        self.buffer = self.buffer[:24]
    self.paint()
  # key()


  def paint(self):
    if self.delete:
      line = "DELETE".ljust(24)
    else:
      line = self.buffer.ljust(24)[:24]
    self.app.mcdu.paint(13, 0, "large", "white", line)
  # paint()

# ScratchPad


class AOC:
  """AOC Page Handlers. There are common things and MCDU-dependent things."""

  # Class variables: shared between all MCDU heads.
  head_list = []
  callsign = None
  received_messages = []


  @classmethod
  def onUplink(cls, frm, typ, text):
    """This thing receives all message uplinks from the ACARS module.
       Not tech stuff like ping."""
    # CPDLC messages are routed to the FANS subsystem, the rest to AOC.
    if typ=="cpdlc":
      FANS.from_acars(frm, text)
    else:
      AOC.received_messages.append(Message(frm, typ, text))
      AOC.event("new_message")
      psx.send("AcarsMsg", "1")
  # onUplink()


  @classmethod
  def event(cls, key, value=None):
    """This receives all events internal to the AOC class and dispatches
       them to all currently active page handlers."""
    for head in AOC.head_list:
      # We should know that the MCDU is on hold.
      # TODO psx.py sends MCDU events "logon", "hold" and "resume". Use them?
      if head.mcdu.headState=="active":
        try:
          head.curr_page(key, value)
        except Exception as ex:
          # This is the main entry point of a whole lot of handlers, and if
          # this blows up, many things will quietly fail. So catch it but
          # make some noise.
          print(f"\n{rd}[{key}({value})] {ex}{yl}")
          traceback.print_exception(type(ex), ex, ex.__traceback__)
          print(df)
          psx.send("FreeMsgS", "DATALINK SYS")
  # event()


  def __init__(self, acars):
    """Add a new AOC page handler to the system. It gets the connection to
       Hoppie's ACARS as parameter."""
    self.acars = acars
    self.mcdu  = None       # MCDU head needs to be set later (chicken, egg)
    self.curr_page = self.home
    self.sub_page  = 1      # 1/n, for subpages.
    self.max_page  = 1      #   n
    self.sp    = ScratchPad(self)
    AOC.head_list.append(self)
  # init()


  def __del__(self):
    """Destructor for an AOC page set."""
    AOC.head_list.remove(self)
  # del()


  def home(self, key="init", value=None):
    """Main ACARS home page. Other name?"""
    if key=="init":
      # Entering the page.
      self.curr_page = self.home
      self.mcdu.clear()
      self.mcdu.paint(0, 0, "large", "white",
                      "HACARS INIT".center(24))
      self.mcdu.paint(1, 0, "small", "white", " FLT ID")
      if AOC.callsign is None:
        fltno = psx.get("FmcFltNo")
        if fltno!='-':
          AOC.callsign = fltno
      if AOC.callsign is None:
        self.mcdu.paint(2, 0, "large", "amber", "bbbbbbb")
      else:
        # TODO need AOC boolean for established?
        self.mcdu.paint(2, 0, "large", "green", AOC.callsign)
      self.mcdu.paint(6, 16, "large", "white", "REQUEST>")
      AOC.event("new_message")

    elif key=="datalink_established":
      self.mcdu.paint(2, 0, "large", "green", AOC.callsign)

    elif key=="new_message":
      # Re-paint the whole message selector block.
      new = 0
      for m in AOC.received_messages:
        if m.is_new: new += 1
      if new>0:
        t = f"{new:3} NEW"
        self.mcdu.paint(11, 17, "small", "amber", t)
        self.mcdu.paint(12, 15, "large", "amber", "MESSAGES>")
      elif len(AOC.received_messages)>0:
        t = f"{len(AOC.received_messages):3} OLD"
        self.mcdu.paint(11, 17, "small", "cyan", t)
        self.mcdu.paint(12, 15, "large", "cyan", "MESSAGES>")
      else:
        self.mcdu.paint(11, 17, "small", "white", "     NO")
        self.mcdu.paint(12, 15, "large", "white", "MESSAGES>")

    elif key=="1L":
      # Flight ID key.
      if self.sp.delete:
        AOC.callsign = None
        self.sp.key("DEL")
        self.mcdu.paint(2, 0, "large", "amber", "bbbbbbb")
      elif self.sp.buffer!="":
        # Upselect sp into FLT ID.
        # TODO This still accepts a lot of nonsense.
        AOC.callsign = self.sp.buffer[:7]
        self.sp.key("CLR+")
        self.mcdu.paint(2, 0, "large", "amber", AOC.callsign.ljust(7))
        # TODO Sync AOC and ATS ACARS?
        print(f"{mg}Need to update FMC Flight Number{df}")
      self.acars.set_callsign(AOC.callsign)
        
      # To perform a quick link check, execute a server ping.
      # The ACARS library sets the connection timeout to 2 seconds.
      self.ping = asyncio.create_task(self.acars.ping_server())
      modules.acars.after(3, lambda: self.pong())

    elif key=="3R":
      # Request key.
      self.request()

    elif key=="6R":
      # Messages key.
      self.index()
  # home()


  def index(self, key="init", value=None):
    """Received message index, last received message on top."""

    if key=="init":
      # Entering the page.
      self.curr_page = self.index
      self.sub_page  = 1
      self.asked_confirmation = False
      self.mcdu.clear()
      self.mcdu.paint(0, 0, "large", "white",
                      "HACARS MSG INDEX".center(20))
      self.mcdu.paint(12, 0, "large", "white", "<RETURN")
      self.mcdu.paint(11,16, "small", "white", "ALL MSGS")
      self.mcdu.paint(12,17, "large", "white", "DELETE>")
      AOC.event("new_message")

    elif key=="new_message":
      if len(AOC.received_messages)==0:
        # No messages, so just an empty page.
        self.mcdu.paint( 0,19, "small", "white", " "*5)
        self.mcdu.paint( 2, 0, "small", "white", "NO MESSAGES".center(24))
        self.mcdu.paint( 4, 0, "small", "white", " "*24)
        self.mcdu.paint( 6, 0, "small", "white", " "*24)
        self.mcdu.paint( 8, 0, "small", "white", " "*24)
        self.mcdu.paint(10, 0, "small", "white", " "*24)
        return

      # Update current subpage/maxpage numbers.
      self.max_page = ceil(len(AOC.received_messages)/5)
      self.mcdu.paint(0, 19, "small", "white",
                      f"{self.sub_page:2}/{self.max_page}")
      # Repaint the whole message list in reverse order.
      msg_nr = len(AOC.received_messages)-((self.sub_page-1)*5)-1
      # Paint MCDU lines 2, 4, 6, 8, 10.
      for line_nr in range(2,10,2):
        if msg_nr>=0:
          # Paint the message signature padded with spaces to the right.
          msg = AOC.received_messages[msg_nr]
          zulu = msg.zulu.strftime("%H%M")
          typp = msg.type.upper()
          orig = msg.orig.upper()
          if msg.is_new:
            self.mcdu.paint(line_nr, 0, "large", "amber",
                            f"<{zulu}  * {typp:5} {orig[:9]}".ljust(24))
            self.mcdu.paint(line_nr, 5, "small", "amber", f"Z")
          else:
            self.mcdu.paint(line_nr, 0, "large", "white",
                            f"<{zulu}    {typp:5} {orig[:9]}".ljust(24))
            self.mcdu.paint(line_nr, 5, "small", "white", f"Z")
        else:
          # Erase line completely.
          self.mcdu.paint(line_nr, 0, "small", "white", " "*24)
        msg_nr -= 1
      # for 5 messages

    elif key in ["1L","2L","3L","4L","5L"]:
      # 1L is last received message, etc.
      m = int(key[0])
      # Display the selected message number.
      self.message("init", len(AOC.received_messages)-m)

    elif key=="6L":
      # Return key.
      self.home()

    elif key=="6R":
      # Delete all received messages. Ask confirmation first.
      if not self.asked_confirmation:
        self.mcdu.paint(11,16, "small", "white", "  DELETE")
        self.mcdu.paint(12,16, "large", "white", "CONFIRM>")
        self.asked_confirmation = True
      else:
        AOC.received_messages = []
        self.mcdu.paint(11,16, "small", "white", "ALL MSGS")
        self.mcdu.paint(12,16,"large", "white", " DELETE>")
        self.asked_confirmation = False
        AOC.event("new_message")

    elif key=="NEXT":
      self.sub_page += 1
      if self.sub_page>self.max_page:
        self.sub_page = 1
      AOC.event("new_message")

    elif key=="PREV":
      self.sub_page -= 1
      if self.sub_page<1:
        self.sub_page = self.max_page
      AOC.event("new_message")
  # index()


  def request(self, key="init", value=None):
    """Request selection page."""

    if key=="init":
      # Entering the page.
      self.curr_page = self.request
      self.mcdu.clear()
      self.mcdu.paint(0, 0, "large", "white",
                      "HACARS REQUEST".center(24))
      if AOC.callsign is None:
        self.mcdu.paint(2,0, "small", "white", "ACARS NOT ACTIVE".center(24))
      else:
        # Just a simple demo.
        self.weather_icao = None
        self.mcdu.paint(3,16,"small","white","AIRPORT")
        self.mcdu.paint(4,19,"large","amber","bbbb")
        self.mcdu.paint(5,16,"small","white","WEATHER")
        self.mcdu.paint(6,15,"large","white","REQUEST>")
      self.mcdu.paint(12, 0, "large", "white", "<RETURN")

    elif key=="2R":
      # Airport ICAO code key.
      if self.sp.buffer!="":
        self.weather_icao = self.sp.buffer[:4]
        self.sp.key("CLR+")
        self.mcdu.paint(4,19,"large","white",self.weather_icao)

    elif key=="3R":
      # Airport Weather Request key.
      if self.weather_icao is not None:
        modules.acars.after(2, lambda:
          self.acars.downlink_inforeq("metar", self.weather_icao))
        modules.acars.after(2, lambda:
          self.mcdu.paint(4,19,"large","green",self.weather_icao))

    elif key=="6L":
      # Return key.
      self.home()
  # request()


  def message(self, key="init", value=None):
    """value = On init, the message number to display."""

    if key=="init" and value is None:
      raise ValueError("cannot display unnumbered message")

    # TODO when this message is deleted by somebody else, we really should
    # exit this page completely and not hang around with a retained but now
    # obsolete/stale set of lines to browse through.
    # One way is to introduce a new event: delete_message.

    if key=="init":
      # Entering the page.
      self.curr_page = self.message
      self.sub_page  = 1
      try:
        msg = AOC.received_messages[value]
      except IndexError:
        # Message was probably deleted by somebody else. Return to index.
        print(f"Message {value} does not exist")
        self.index()
        return
      print(f"Displaying message {value}")

      # This code retains line feeds and wraps overlong lines, but does not
      # retain empty lines. For MCDU display this is okay.
      lines = msg.text.splitlines()
      wrapped_lines = [tw.wrap(line, 24) for line in lines]
      self.mcdu_lines = [item for sublist in wrapped_lines for item in sublist]
      self.max_page = ceil(len(self.mcdu_lines)/9)

      if msg.is_new:
        msg.is_new = False
        psx.send("AcarsMsg", "0")
        AOC.event("new_message")

      self.mcdu.clear()
      zulu = msg.zulu.strftime("%H%M")
      self.mcdu.paint( 0, 0, "large", "white", f"  HACARS MSG {zulu}Z")
      self.mcdu.paint(12, 0, "large", "white", "<INDEX")
      AOC.event("message_repaint")

    elif key=="message_repaint":
      # Update current subpage number.
      self.mcdu.paint(0, 19, "small", "white",
                      f"{self.sub_page:2}/{self.max_page}")
      # Paint MCDU lines 2..11.
      mcdu_line_nr = 2
      msg_line_nr  = (self.sub_page-1)*9
      for mcdu_line_nr in range(2,11):
        if msg_line_nr<len(self.mcdu_lines):
          # Paint padded with spaces to the right.
          self.mcdu.paint(mcdu_line_nr, 0, "small", "white",
                          self.mcdu_lines[msg_line_nr].ljust(24));
        else:
          # Erase line completely.
          self.mcdu.paint(mcdu_line_nr, 0, "small", "white", " "*24)
        msg_line_nr  += 1

    elif key=="6L":
      # Return key.
      self.index()

    elif key=="NEXT":
      self.sub_page += 1
      if self.sub_page>self.max_page:
        self.sub_page = 1
      AOC.event("message_repaint")

    elif key=="PREV":
      self.sub_page -= 1
      if self.sub_page<1:
        self.sub_page = self.max_page
      AOC.event("message_repaint")
  # message()


  def pong(self):
    """Check whether the datalink actually worked."""
    if not self.ping.done():
      # The HTTP attempt really should have been completed/rejected by now.
      print("PING NOT DONE YET, PANIC!")
    else:
      r = self.ping.result()
      if r.startswith("ok"):
        # Extinghuish all EICAS and kick the subsystem into life.
        print("Datalink established")
        psx.send("FreeMsgC", "")
        psx.send("FreeMsgA", "")
        psx.send("FreeMsgM", "")
        psx.send("FreeMsgS", "")
        AOC.event("datalink_established")
      else:
        # Let the crew know it's not okay.
        r = r[7:-1]
        print("Datalink NOT established:", r)
        # Upper EICAS gets a general caution,
        psx.send("FreeMsgC", "DATALINK SYS")
        psx.send("FreeMsgA", ">NO COMM")
        psx.send("FreeMsgM", "")
        # Lower EICAS gets more detail. Max 16 chars for STATUS messages.
        if r.startswith("Cannot connect"):
          psx.send("FreeMsgS", "SERVER UNKNOWN")
        elif r.startswith("Connection timeout"):
          psx.send("FreeMsgS", "CONN TIMEOUT")
        elif r.startswith("ACARS server unreachable"):
          psx.send("FreeMsgS", "SERVER UNAVAIL")
        elif r.startswith("invalid logon code"):
          psx.send("FreeMsgS", "INVLD LOGON CODE")
        elif r.startswith("callsign already in use"):
          psx.send("FreeMsgS", "DUPL CALLSIGN")
        else:
          psx.send("FreeMsgS", "ACARS FAULT")
    self.ping = None
  # pong()


  def mcduEvent(self, mcdu, type, value=None):
    """ Called by an MCDU when it has something to report or request. """
    # TODO the head should start as logged off. There are events for this,
    # such as "hold" and "logoff". Only when "logon" has been received the
    # head should be registered active by the AOC handler.
    if type in ["logon", "resume"]:
      self.curr_page()
    elif type=="keypress":
      if value in ["1L","2L","3L","4L","5L","6L",
                   "1R","2R","3R","4R","5R","6R",
                   "PREV","NEXT"]:
        self.curr_page(value)
        # Maybe this is not truly realistic, but it just feels better.
        self.mcdu.blank()
      elif value in ["INITREF","RTE","DEPARR","ATC","VNAV","FIX",
                     "LEGS","HOLD","FMCCOMM","PROG","EXEC","NAVRAD",
                     "ATC+","REL"]:
        pass
      else:
        # Alphanumerical keys and a few symbols all go to the scratch pad.
        self.sp.key(value)
    else:
      print(f"Unhandled MCDU event from {mcdu.location}: {type}={value}")
  # mcduEvent()

# AOC


# Don't want to have to type FANS.Up_Down every time.
# These are PSX enumerations, not necessarily in line with Hoppie's ACARS
# conventions.
class Up_Down(Enum):
  DOWNLINK = "D"
  UPLINK   = "U"
class Msg_Type(Enum):
  EMERGENCY = "0"
  REPORT    = "1"
  REQUEST   = "2"
class Status(Enum):
  NEW      = "0"
  OPEN     = "1"
  STANDBY  = "2"
  SENT     = "3"
  ACCEPTED = "4"
  REJECTED = "5"
  RESP_RXD = "6"
  OLD      = "7"
class Resp_Rq(Enum):
  NA    = "0"  # Not Applicable, no response required  TODO NE?
  ROGER = "1"  # Roger required
  AN    = "2"  # Affirm/Negative required
  WU    = "3"  # Wilco/Unable required


class FANS():
  """Not very well organized yet, use this for grouping purposes.
     https://aerowinx.com/board/index.php/topic,6944.msg74832.html
  """

  curr_atc = None
  next_atc = None


  def __init__(self, send_id=None, resp_id=None, up_down=None, typ=None,
                     status=None,  resp_rq=None, text=None):
    """FANS message."""
    self.send_id = send_id
    self.resp_id = resp_id
    self.up_down = up_down
    self.typ     = typ
    self.status  = status
    self.resp_rq = resp_rq
    self.text    = text
  # init()


  def __str__(self):
    """Display of a FANS message. Good for debugging and monitoring."""
    s =  f"Send ID: {self.send_id}\n"
    s += f"Resp ID: {self.resp_id}\n"   # -1 = does not respond to anything
    s += f"Up/Down: {self.up_down.name}\n"
    s += f"Type:    {self.typ.name}\n"
    s += f"Status:  {self.status.name}\n"
    s += f"Resp Rq: {self.resp_rq.name}\n"
    s += f"Text:    {self.text}"        # list of string elements
    return s
  # str()


  def from_psx(key, value):
    """Process a FansDn or FansDnResp PSX event. Returns a FANS object (msg)."""
    # FansUp/FansUpResp messages may come in from PSX if the manual CPDLC
    # instructor has been made active and is sending stuff. This is good for
    # testing.
    print(f"{cy}  psx: {key} {value}{df}")

    if key=="FansDn" or key=="FansUp":
      # 447;-1;0;1397652317;U;0;0;3;/DESCEND TO^FL340^;
      m = FANS()
      fields = value.split(";")
      m.send_id = fields[0]
      m.resp_id = fields[1]
      # ignore fields[2], response mode. Typically 0. TODO this probably is it.
      m.time    = fields[3] # seconds since epoch
      m.up_down = Up_Down(fields[4])
      m.typ     = Msg_Type.REQUEST  # TODO emergencies and reports
      m.status  = Status(fields[6])
      m.resp_rq = Resp_Rq(fields[7])
      m.text = []
      for f in fields[8].split("^")[:-1]:
        m.text.append(f.strip())
      print(f"From PSX:\n{yl}{m}{df}")
      # On the wire:
      # /data2/2//Y/WHEN CAN WE EXPECT@@CLIMB TO@FL370
      # AcarsAir:
      # /data2/3//Y/WHEN CAN WE EXPECT@@CLIMB TO@FL370@|FREE TEXT@TEST TEST@@TEST 2
      # PSX:
      # [' WHEN CAN WE EXPECT', '', ' CLIMB TO', 'FL370']
      # [' REQUEST CLIMB TO', 'FL370']
      # TODO PSX prefixes free text with "/FREE TEXT" which includes a false
      # delimiter "/".
      if m.up_down==Up_Down.DOWNLINK:
        text = "@".join(m.text)
        data = f"/data2/{m.send_id}/{m.resp_id}/Y/{text}"
        acars.downlink_cpdlc(FANS.curr_atc, data)
      return m
    elif key=="FansDnResp":
      # 27;2;1397653075;;
      fields  = value.split(";")
      resp_to = fields[0]   # MRN
      free    = ""
      if fields[1]=="-3":
        # Reject due to performance.
        resp = "reject"
        free = "due to performance"
      elif fields[1]=="-2":
        # Reject due to weather.
        resp = "reject"
        free = "due to weather"
      elif fields[1]=="-1":
        # Reject.
        resp = "reject"
      elif fields[1]=="0":
        # Blank.
        resp = ""
      elif fields[1]=="1":
        # Standby.
        resp = "standby"
      elif fields[1]=="2":
        # Wilco.
        resp = "wilco"
      elif fields[1]=="3":
        # Affirm.
        resp = "affirm"
      elif fields[1]=="4":
        # Roger
        resp = "roger"
      else:
        # ???
        resp = "duh"
      epoch = fields[2]
      if fields[3]!="":
        free += "^"+fields[3]  # free^^text^^
      # MsgID is 0 because PSX does not consider responses to be new messages.
      # Expected response type is N as this is already a response.
      data = f"/data2/0/{resp_to}/N/{resp} {free}"
      acars.downlink_cpdlc(FANS.curr_atc, data.strip())
    elif key=="FansUpResp":
      # Only used for testing by PSX manual Instructor CPDLC.
      pass
  # from_psx()


  def from_acars(frm, data):
    #TODO best to have a boolean "awaiting logon acceptance" because if ATC
    # does not accept, we need to yell something at the pilots.
    if "LOGON ACCEPTED" in data:
      FANS.accept()
    elif data.startswith("/data2/"):
      m = FANS()
      fields = data.split("/")
      m.send_id = fields[2]
      if fields[3]=="":
        m.resp_id = -1
      else:
        m.resp_id = fields[3]
      m.up_down = Up_Down.UPLINK
      m.typ     = Msg_Type.REQUEST    # TODO this is not always the case
      m.status  = Status.NEW          # TODO this is not always the case
                                      # If resp_id then this is RESP_RXD
      if fields[4]=="NE":
        m.resp_rq = Resp_Rq.NA
      elif fields[4]=="R":
        m.resp_rq = Resp_Rq.ROGER
      elif fields[4]=="WU":
        m.resp_rq = Resp_Rq.WU
      elif fields[4]=="AN":
        m.resp_rq = Resp_Rq.AN
      # When CPDLC sends chunks that exceed 24 chars, which should not
      # happen but people don't seem to care, PSX cuts off everything beyond
      # the 24th character.
      text = [tw.wrap(line, 24) for line in fields[5].split("@")]
      # This list of lists needs to be unwrapped one level.
      m.text = [item for sublist in text for item in sublist]
      print(f"From ACARS:\n{yl}{m}{df}")

      if FANS.curr_atc is None:
        print(f"FANS is logged off, uplink from {frm} ignored")
        return

      # First let PSX know this uplink is a response uplink.
      # 3 = "affirm", we don't do deep packet inspection here.
      # TODO But ATC can also uplink originals, not responses? Yes but this
      # seems not a problem at all for this code. Works fine.
      # TODO ATC can deny a downlink request but ack its reception. How?
      epoch = int(time.time())
      ru = f"{m.resp_id};{m.send_id};3;{epoch};"
      psx.send("FansUpResp", ru)
      time.sleep(1)

      # Now create and send the FansUp.
      data = "^".join(m.text)
      epoch = int(time.time())
      fu = f"{m.send_id};{m.resp_id};0;{epoch};{m.up_down.value};" \
           f"2;{m.status.value};{m.resp_rq.value};{data}^;"
      psx.send("FansUp", fu)
    else:
      printf(f"FANS: ignored invalid data format (must be data2)")
  # to_psx()


  def FansBasics(key, value):
    """PSX doc describes how to handle this long string variable:
       https://aerowinx.com/board/index.php/topic,6944.msg74832.html"""
    print(f"{cy}  psx: {key} {value}{df}")
    fields = value.split(";")
    timer = fields[0]
    logon_to = fields[1]
    curr_atc = fields[2]
    if timer=="0":    # "logged off"
      # Only do something if we know we are logged on.
      if FANS.curr_atc is not None:
        print(f"{yl}FANS: logging off{df}")
        acars.downlink_cpdlc(FANS.curr_atc, "/data2/1//Y/LOG OFF")
        FANS.logoff()

    elif timer=="6":    # "sent"
      print(f"{yl}FANS: logon_to:[{logon_to:4}] curr_atc:[{curr_atc:4}]{df}")
      # TODO
      # This is crude. Do we need both AOC and Client classes to keep the
      # callsign? Can AOC not proxy for Client, or AOC drop it entirely?
      fltno = psx.get("FmcFltNo")
      AOC.callsign = fltno
      acars.set_callsign(fltno)
      acars.downlink_cpdlc(logon_to, "/data2/1//Y/REQUEST LOGON")

    elif timer=="10":
      print(f"{mg}Timer==10 in FansBasics, did not expect this.{df}")

    elif timer=="11":
      # Reached conclusion of logon process. But this may be a simulated
      # logoff.
      print(f"{yl}FANS: logon_to:[{logon_to:4}] curr_atc:[{curr_atc:4}]{df}")
      if curr_atc=="":
        print("FANS is logged off")
        FANS.curr_atc = None
      else:
        print(f"FANS is logged on to {curr_atc}")
        FANS.curr_atc = curr_atc
        # Extinguish ">NO COMM" if it was still there (no AOC done yet).
        psx.send("FreeMsgA", "")
  # FansBasics()


  def accept():
    # This is a PSX-specific action, PSX native does not understand the
    # concept of "logon accepted" as it always simulates an acceptation.
    print("FANS logon accepted by ATC")
    fb = psx.get("FansBasics")
    fb = FANS.replace(fb, 0, "8")
    psx.send("FansBasics", fb)  # bump the counter to 8 to advance the FSM
  # accept()


  def reject():
    print("FANS logon rejected by ATC")
    FANS.logoff()
  # reject()


  # Simple function to replace a ;; field in a string.
  # TODO Should probably end up in the PSX module.
  def replace(q_string, field_no, replacement):
    fields = q_string.split(";")
    fields[field_no] = replacement
    return ";".join(fields)
  # replace()


  def handoff():
    # TODO
    print(f"{mg}Handoff is TODO{df}")
    """A few options work. Just replace the "ACT ATC" field.
       "NEXT CTR" is always blank by design.
    1. Timer 1 with a new callsign in field[1]. Retain field[2] so that the
       ACT CTR does not drop away until it has been accepted.
    2. Timer 8 with same setup as 1. This is less noisy.
    3. Did not yet try flipping field [2] without anything else.
    """
  # handoff()


  def logoff():
    print("FANS log off")
    fb = psx.get("FansBasics")
    # Timer==10 makes it look like we logged off, ATC CTR blanks, but PSX
    # still thinks we're logged on and will send requests to a blank call
    # sign.
    fb = FANS.replace(fb, 0, "10")
    fb = FANS.replace(fb, 1, "")
    fb = FANS.replace(fb, 2, "")
    psx.send("FansBasics", fb)
  # logoff()

# FANS


def set_zulu(key, time_earth):
    """Remember the PSX planet time (not necessarily real UTC)."""
    global ZULU

    epoch = int(time_earth)/1000    # PSX thinks in milliseconds
    first = (ZULU is None)
    ZULU = dt.fromtimestamp(epoch, timezone.utc)
    if first:
      print(f"{cy}  psx: simulator time is {ZULU}{df}")
# set_utc()


def bg_exception(loop, context):
  """The background exception handler is used to catch things that go wrong
     on background tasks (such as created with "after") that have no natural
     call stack to backtrack looking for a handler. If this background
     exception handler is not installed, an additional "Task exception was
     never retrieved" usually is the result.
  """
  ex = context.get("exception")
  print(f"\n{rd}[BG] {ex}{yl}")
  traceback.print_exception(type(ex), ex, ex.__traceback__)
  print(df)
  psx.send("FreeMsgS", "DATALINK SYS")
# bg_exception()


async def supervisor():
  """This is the inittab, so to say."""
  loop = asyncio.get_event_loop()
  loop.set_exception_handler(bg_exception)
  try:
    await asyncio.gather(
      psx.connect(),
      acars.install(AOC.onUplink),
    )
  except Exception as ex:
    # This is the emergency exit.
    print(f"\n{rd}{ex}{yl}")
    traceback.print_exception(type(ex), ex, ex.__traceback__)
    print(df)
    teardown("supervisor exception")
    # Give the system a second to cleanly shut down and then just quit.
    await asyncio.sleep(1)
    os._exit(1)
# supervisor()


##### MAIN ###############################################################

# To be fancy, we use colours. Windows terminals understand ANSI by now.
df = "\033[0m"  # default colour
bk = "\033[30m"
rd = "\033[31m"
gr = "\033[32m"
yl = "\033[33m"
bl = "\033[34m"
mg = "\033[35m"
cy = "\033[36m"
wh = "\033[37m"

print(f"{df}Hoppie's ACARS client for Aerowinx PSX, version {VERSION}")

# Temporary: get the locon code as the first command line parameter.
if len(sys.argv)<2:
  print("Need logon code as command line parameter")
  exit(1)
logon_code = sys.argv[1]

# Stealth mode uses PEEK instead of POLL.
stealth = (len(sys.argv)==3 and sys.argv[2]=="--stealth")
if stealth:
  print(f"Running in Stealth Mode, CPDLC only, {yl}requires BACARS to run{df}")

with modules.psx.Client() as psx:
  # psx.logger = lambda msg: print(f"{cy}  psx: {msg}{df}")

  psx.onResume     = setup
  psx.onPause      = teardown
  psx.onDisconnect = teardown

  psx.subscribe("FansBasics", FANS.FansBasics)
  psx.subscribe("FansDn", FANS.from_psx)
  psx.subscribe("FansUp", FANS.from_psx)
  psx.subscribe("FansDnResp", FANS.from_psx)
  psx.subscribe("FansUpResp", FANS.from_psx)
  psx.subscribe("FansCont")
  psx.subscribe("FansAtcMsgId", lambda key, value:
    print(f"{cy}  psx: new AIC Msg Id {value}{df}"))
  psx.subscribe("FmcFltNo")
  psx.subscribe("FmcRte1")  # split(;) [0]=orig [1]=dest default "bbbb"
  psx.subscribe("FmcRte2")
  psx.subscribe("id")
  psx.subscribe("version", lambda key, value:
    print(f"{cy}  psx: connected to PSX {value} as client "
          f"#{psx.get('id')}{df}"))
  psx.subscribe("TimeEarth", set_zulu)

  # Acquire an ACARS subsystem, but don't install it yet.
  acars = modules.acars.Client(logon_code, stealth_mode = stealth)
  acars.logger = lambda msg: print(f"{gr}  acars: {msg}{df}")

  if not stealth:
    # Wire the AOC page set to all three MCDUs, at L2.
    acars_l = AOC(acars)
    mcdu_l = modules.psx.MCDU("L", "L", 2, "<HACARS", acars_l.mcduEvent)
    acars_l.mcdu = mcdu_l
    acars_c = AOC(acars)
    mcdu_c = modules.psx.MCDU("C", "L", 2, "<HACARS", acars_c.mcduEvent)
    acars_c.mcdu = mcdu_c
    acars_r = AOC(acars)
    mcdu_r = modules.psx.MCDU("R", "L", 2, "<HACARS", acars_r.mcduEvent)
    acars_r.mcdu = mcdu_r

  try:
    asyncio.run(supervisor())
  except KeyboardInterrupt:
    print("\nStopped by keyboard interrupt (Ctrl-C)")

# EOF
